/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.ComponentModel;
using System.ComponentModel.Design.Serialization;// for type converter
using System.Globalization; // for type converter, CultureInfo
using System.Reflection;// for type converter
using System.Drawing.Design; // for type editor

using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.UmlRt;
using Borland.Eco.Globalization;

namespace Borland.Eco.Handles
{

	// Type converter to make Nesting Serialize
	public sealed class NestingConverter: ExpandableObjectConverter
	{
		public override bool CanConvertTo(ITypeDescriptorContext context, Type destinationType)
		{
			if (destinationType == typeof(InstanceDescriptor))
				return true;
			return base.CanConvertTo(context, destinationType);
		}

		public override object ConvertTo(ITypeDescriptorContext context, CultureInfo culture, object value, Type destinationType)
		{

			if (destinationType == typeof(InstanceDescriptor))
			{
				if (value != null)
				{
					ConstructorInfo ctor = typeof(Nesting).GetConstructor(
						new Type[0] {});

					return new InstanceDescriptor(ctor,
						new object[0]{},false);
				}
			}
			return base.ConvertTo(context, culture, value, destinationType);
		}
	}

	[Serializable()]
	[TypeConverter(typeof(NestingConverter))]
	public sealed class Nesting: IStaticContext, IHasTypeSystem
	{
		private NestingCollection m_Owner;
		private string m_Name = "";
		private string m_TypeName = "";

		/// <summary>
		/// For internal use.
		/// </summary>
		///<exception cref="InvalidOperationException">Thrown if the new nesting has no owner.</exception>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public IStaticContext RootContext
		{
			get 
			{ 
				if (m_Owner != null)
					return m_Owner.StaticContext; 
				else
					throw new InvalidOperationException(HandlesStringRes.sOwnerNotSet);
			}
		}
		internal IStaticContext AssertedRootContext
		{
			get
			{
				IStaticContext result = RootContext;
				if (result != null)
					return result;
				else
					throw new InvalidOperationException(HandlesStringRes.sRootContextNotSet);
			}
		}				

		public Nesting() {}

		internal NestingCollection Owner
		{
			set
			{
				if ((m_Owner != null) && (m_Owner != value))
					throw new NotSupportedException();
				m_Owner = value;
			}
		}

		private void Changed()
		{
			if (m_Owner != null)
				m_Owner.Changed();
		}

		private void ColumnsChanged(System.Object dummy1, EventArgs dummy2)
		{
			Changed();
		}

		private ColumnCollection m_Columns;

		/// <summary>
		/// See ElementHandle.Columns.
		/// </summary>
		[Editor("CollectionEditor", "UITypeEditor")]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyColumns")]
		public ColumnCollection Columns
		{
			get
			{
				if ( m_Columns == null)
				{
					 m_Columns = new ColumnCollection(this);
					 m_Columns.OnChange += new EventHandler(ColumnsChanged);
				}
				return m_Columns;
			}
		}

		private bool m_AddDefaultProperties = true;

		/// <summary>
		/// See ElementHandle.AddDefaultProperties.
		/// </summary>
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyAddDefaultProperties")]
		[DefaultValue(true)]
		public bool AddDefaultProperties
		{
			get { return m_AddDefaultProperties; }
			set { m_AddDefaultProperties = value; }
		}

		private bool m_AddDefaultNestings = true;

//  FIXME		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyAddDefaultProperties")]
		/// <summary>
		/// See ElementHandle.AddDefaultNestings.
		/// </summary>
		[DefaultValue(true)]
		public bool AddDefaultNestings
		{
			get { return m_AddDefaultNestings; }
			set { m_AddDefaultNestings = value; }
		}
		private void CheckUniqueName(string name)
		{
			if (m_Owner == null) return;
			if (!m_Owner.NameIsUnique(name))
				throw new InvalidOperationException(HandlesStringRes.sNameNotUnique(name));			
		}
		/// <summary>
		/// This name is refered to by Column.NestingName.
		/// </summary>
		///<exception cref="InvalidOperationException">Thrown if the new name is not unique in the owning collection.</exception>
		public String Name
		{
			get { return m_Name; }
			set
			{
				if (m_Name != value)
				{
					NameValidator.AssertNameIsValid(value);
					CheckUniqueName(value);
					m_Name = value;
					Changed();
				}
			}
		}

		/// <summary>
		/// The type of the objects in the list the nesting applies to.
		/// </summary>
		[Editor("Borland.Eco.Handles.Design.NonCollectionTypeNameSelectorEditor, Borland.Eco.Handles.Design", typeof(UITypeEditor))]
		public string TypeName
		{
			get { return m_TypeName; }
			set
			{
				if (m_TypeName != value)
				{
					m_TypeName = value;
					Changed();
				}
			}
		}

#region IHasTypeSystem
		IEcoTypeSystem IHasTypeSystem.TypeSystem
		{
			get { return AssertedRootContext.TypeSystem; }
		}
		bool IHasTypeSystem.IsHookedUp { get { return RootContext != null && AssertedRootContext.TypeSystem != null; } }
#endregion

		IEcoTypeSystem IStaticContext.TypeSystem
		{
			get { return AssertedRootContext.TypeSystem; }
		}

		/// <summary>
		/// For internal use.
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public IOclTypeService OclTypeService  // Doubles for IStaticContext
		{
			get { return AssertedRootContext.OclTypeService; }
		}

		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public IOclPsTypeService OclPsTypeService  // Doubles for IStaticContext
		{
			get { return AssertedRootContext.OclPsTypeService; }
		}

		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public IActionLanguageTypeService ActionLanguageTypeService  // Doubles for IStaticContext
		{
			get { return AssertedRootContext.ActionLanguageTypeService; }
		}

		/// <summary>
		/// For internal use.
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public IExternalVariableList VariableList  // Doubles for IStaticContext
		{
			get { return AssertedRootContext.VariableList; }
		}

		IClassifier IStaticContext.StaticUmlType
		{
			get
			{
				if (AssertedRootContext.TypeSystem != null)
					try
					{
						return AssertedRootContext.TypeSystem.GetClassifierByName(TypeName);
					}
					catch
					{
						// Ignore errors, just return null type.
					}
				return null;
			}
		}
	}

	//NestingCollection
	[Serializable()]
	[ListBindable(false)]
	public sealed class NestingCollection: System.Collections.CollectionBase
	{
		private IStaticContext m_StaticContext;
		public IStaticContext StaticContext
		{
			get { return m_StaticContext; }
		}

		public NestingCollection(IStaticContext staticContext): base()
		{
			m_StaticContext = staticContext;
		}

		public Nesting this[int index]
		{
			get { return (Nesting) List[index]; }
		}

		public Nesting this[string name]
		{
			get
			{
				foreach (Nesting n in List)
					if (n.Name == name)
						return n;
				return null;
			}
		}

		public void Add(Nesting nesting)
		{
			List.Add(nesting);
		}

		public int IndexOf(System.Object value)
		{
			return List.IndexOf(value);
		}

		public void AddRange(Nesting[] nestings)
		{
			foreach(Nesting aNesting in nestings)
			{
				Add(aNesting);
			}
		}

		internal Boolean NameIsUnique(String name)
		{
			for (int i = 0; i < Count; i++)
				if (string.Compare(name, this[i].Name, true, CultureInfo.InvariantCulture) == 0) // case insensitive
					return false;
			return true;
		}

		private string GetUniqueName()
		{
			int i;
			String value;

			i = 1;
			do
			{
				value = "New" + i.ToString(CultureInfo.InvariantCulture); // Do not localize
				i++;
			} while (!NameIsUnique(value));
			return value;
		}

		internal void Changed()
		{
			if (OnChange != null)
				OnChange(this, EventArgs.Empty);
		}

		///<exception cref="InvalidOperationException">Thrown if the inserted nesting's name is not unique.</exception>
		protected override void OnInsert(int index, object value)
		{
			base.OnInsert(index, value);

			Nesting aNesting = (Nesting)value;

			if (aNesting.Name == null || aNesting.Name.Length == 0)
				aNesting.Name = GetUniqueName();
			if (!NameIsUnique(aNesting.Name))
				throw new InvalidOperationException(HandlesStringRes.sNameNotUnique(aNesting.Name));
			aNesting.Owner = this;
		}

		protected override void OnInsertComplete(int index, object value)
		{
			base.OnInsertComplete(index,value);
			Changed();
		}

		protected override void OnClearComplete()
		{
			base.OnClear();
			Changed();
		}

		protected override void OnRemoveComplete(int index, object value)
		{
			base.OnRemoveComplete(index,value);
			Changed();
		}
		public event EventHandler OnChange;
	}
}
